/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.util;

import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.util.HashSet;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DiskChecker {
    private static final Logger LOG = LoggerFactory.getLogger(DiskChecker.class);
    private float diskUsageThreshold;
    private float diskUsageWarnThreshold;

    public DiskChecker(float threshold, float warnThreshold) {
        this.validateThreshold(threshold, warnThreshold);
        this.diskUsageThreshold = threshold;
        this.diskUsageWarnThreshold = warnThreshold;
    }

    private static boolean mkdirsWithExistsCheck(File dir) {
        if (dir.mkdir() || dir.exists()) {
            return true;
        }
        File canonDir = null;
        try {
            canonDir = dir.getCanonicalFile();
        }
        catch (IOException e) {
            return false;
        }
        String parent = canonDir.getParent();
        return parent != null && DiskChecker.mkdirsWithExistsCheck(new File(parent)) && (canonDir.mkdir() || canonDir.exists());
    }

    @VisibleForTesting
    float checkDiskFull(File dir) throws DiskOutOfSpaceException, DiskWarnThresholdException {
        if (null == dir) {
            return 0.0f;
        }
        if (dir.exists()) {
            long totalSpace;
            long usableSpace = dir.getUsableSpace();
            float free = (float)usableSpace / (float)(totalSpace = dir.getTotalSpace());
            float used = 1.0f - free;
            if (used > this.diskUsageThreshold) {
                LOG.error("Space left on device {} : {}, Used space fraction: {} > threshold {}.", new Object[]{dir, usableSpace, Float.valueOf(used), Float.valueOf(this.diskUsageThreshold)});
                throw new DiskOutOfSpaceException("Space left on device " + usableSpace + " Used space fraction:" + used + " > threshold " + this.diskUsageThreshold, used);
            }
            if (used > this.diskUsageWarnThreshold) {
                LOG.warn("Space left on device {} : {}, Used space fraction: {} > WarnThreshold {}.", new Object[]{dir, usableSpace, Float.valueOf(used), Float.valueOf(this.diskUsageWarnThreshold)});
                throw new DiskWarnThresholdException("Space left on device:" + usableSpace + " Used space fraction:" + used + " > WarnThreshold:" + this.diskUsageWarnThreshold, used);
            }
            return used;
        }
        return this.checkDiskFull(dir.getParentFile());
    }

    public long getTotalFreeSpace(List<File> dirs) throws IOException {
        long totalFreeSpace = 0L;
        HashSet<FileStore> dirsFileStore = new HashSet<FileStore>();
        for (File dir : dirs) {
            FileStore fileStore = Files.getFileStore(dir.toPath());
            if (!dirsFileStore.add(fileStore)) continue;
            totalFreeSpace += fileStore.getUsableSpace();
        }
        return totalFreeSpace;
    }

    public long getTotalDiskSpace(List<File> dirs) throws IOException {
        long totalDiskSpace = 0L;
        HashSet<FileStore> dirsFileStore = new HashSet<FileStore>();
        for (File dir : dirs) {
            FileStore fileStore = Files.getFileStore(dir.toPath());
            if (!dirsFileStore.add(fileStore)) continue;
            totalDiskSpace += fileStore.getTotalSpace();
        }
        return totalDiskSpace;
    }

    public float getTotalDiskUsage(List<File> dirs) throws IOException {
        if (dirs == null || dirs.isEmpty()) {
            throw new IllegalArgumentException("list argument of getTotalDiskUsage is not supposed to be null or empty");
        }
        float free = (float)this.getTotalFreeSpace(dirs) / (float)this.getTotalDiskSpace(dirs);
        float used = 1.0f - free;
        return used;
    }

    public float checkDir(File dir) throws DiskErrorException, DiskOutOfSpaceException, DiskWarnThresholdException {
        float usage = this.checkDiskFull(dir);
        if (!DiskChecker.mkdirsWithExistsCheck(dir)) {
            throw new DiskErrorException("can not create directory: " + dir.toString());
        }
        if (!dir.isDirectory()) {
            throw new DiskErrorException("not a directory: " + dir.toString());
        }
        if (!dir.canRead()) {
            throw new DiskErrorException("directory is not readable: " + dir.toString());
        }
        if (!dir.canWrite()) {
            throw new DiskErrorException("directory is not writable: " + dir.toString());
        }
        return usage;
    }

    void setDiskSpaceThreshold(float diskSpaceThreshold, float diskUsageWarnThreshold) {
        this.validateThreshold(diskSpaceThreshold, diskUsageWarnThreshold);
        this.diskUsageThreshold = diskSpaceThreshold;
        this.diskUsageWarnThreshold = diskUsageWarnThreshold;
    }

    private void validateThreshold(float diskSpaceThreshold, float diskSpaceWarnThreshold) {
        if (diskSpaceThreshold <= 0.0f || diskSpaceThreshold >= 1.0f || (double)(diskSpaceWarnThreshold - diskSpaceThreshold) > 1.0E-6) {
            throw new IllegalArgumentException("Disk space threashold: " + diskSpaceThreshold + " and warn threshold: " + diskSpaceWarnThreshold + " are not valid. Should be > 0 and < 1 and diskSpaceThreshold >= diskSpaceWarnThreshold");
        }
    }

    public float getDiskUsageThreshold() {
        return this.diskUsageThreshold;
    }

    public float getDiskUsageWarnThreshold() {
        return this.diskUsageWarnThreshold;
    }

    public static class DiskWarnThresholdException
    extends DiskException {
        private static final long serialVersionUID = -1629284987500841657L;
        private final float usage;

        public DiskWarnThresholdException(String msg, float usage) {
            super(msg);
            this.usage = usage;
        }

        public float getUsage() {
            return this.usage;
        }
    }

    public static class DiskOutOfSpaceException
    extends DiskException {
        private static final long serialVersionUID = 160898797915906860L;
        private final float usage;

        public DiskOutOfSpaceException(String msg, float usage) {
            super(msg);
            this.usage = usage;
        }

        public float getUsage() {
            return this.usage;
        }
    }

    public static class DiskErrorException
    extends DiskException {
        private static final long serialVersionUID = 9091606022449761729L;

        public DiskErrorException(String msg) {
            super(msg);
        }
    }

    public static abstract class DiskException
    extends IOException {
        public DiskException(String msg) {
            super(msg);
        }
    }
}

