/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.spark.utils;

import com.google.common.collect.ImmutableMap;
import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.cassandra.bridge.CassandraBridge;
import org.apache.cassandra.spark.data.CqlTable;
import org.apache.cassandra.spark.data.ReplicationFactor;
import org.apache.cassandra.spark.data.VersionRunner;
import org.apache.cassandra.spark.data.partitioner.Partitioner;
import org.apache.cassandra.spark.utils.CqlUtils;
import org.apache.cassandra.spark.utils.ResourceUtils;
import org.apache.cassandra.spark.utils.TableIdentifier;
import org.apache.commons.io.FileUtils;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;

public class CqlUtilsTest
extends VersionRunner {
    static String fullSchemaSample;
    @TempDir
    private static Path tempPath;

    @BeforeAll
    public static void setup() throws URISyntaxException, IOException {
        fullSchemaSample = CqlUtilsTest.loadFullSchemaSample();
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.cassandra.spark.data.VersionRunner#bridges"})
    public void textExtractIndexCount(CassandraBridge bridge) {
        int indexCount = CqlUtils.extractIndexCount((String)fullSchemaSample, (String)"cycling", (String)"rank_by_year_and_name");
        Assertions.assertThat((int)indexCount).isEqualTo(3);
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.cassandra.spark.data.VersionRunner#bridges"})
    public void testExtractKeyspace(CassandraBridge bridge) {
        String keyspaceSchema = CqlUtils.extractKeyspaceSchema((String)fullSchemaSample, (String)"keyspace");
        String tagEntityRelationV4KeyspaceSchema = CqlUtils.extractKeyspaceSchema((String)fullSchemaSample, (String)"quoted_keyspace");
        String systemDistributedKeyspaceSchema = CqlUtils.extractKeyspaceSchema((String)fullSchemaSample, (String)"system_distributed");
        String systemSchemaKeyspaceSchema = CqlUtils.extractKeyspaceSchema((String)fullSchemaSample, (String)"system_schema");
        Assertions.assertThat((String)keyspaceSchema).isEqualTo("CREATE KEYSPACE keyspace WITH REPLICATION = { 'class' : 'org.apache.cassandra.locator.NetworkTopologyStrategy', 'datacenter1': '4', 'datacenter2': '3' } AND DURABLE_WRITES = true;");
        Assertions.assertThat((String)tagEntityRelationV4KeyspaceSchema).isEqualTo("CREATE KEYSPACE \"quoted_keyspace\" WITH REPLICATION = { 'class' : 'org.apache.cassandra.locator.NetworkTopologyStrategy', 'datacenter1': '3', 'datacenter2': '3' } AND DURABLE_WRITES = true;");
        Assertions.assertThat((String)systemDistributedKeyspaceSchema).isEqualTo("CREATE KEYSPACE system_distributed WITH REPLICATION = { 'class' : 'org.apache.cassandra.locator.SimpleStrategy', 'replication_factor': '3' } AND DURABLE_WRITES = true;");
        Assertions.assertThat((String)systemSchemaKeyspaceSchema).isEqualTo("CREATE KEYSPACE system_schema WITH REPLICATION = { 'class' : 'org.apache.cassandra.locator.LocalStrategy' } AND DURABLE_WRITES = true;");
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.cassandra.spark.data.VersionRunner#bridges"})
    public void testExtractKeyspaceNames(CassandraBridge bridge) {
        Set keyspaceNames = CqlUtils.extractKeyspaceNames((String)fullSchemaSample);
        Assertions.assertThat((Collection)keyspaceNames).hasSize(3);
        Map rfMap = keyspaceNames.stream().collect(Collectors.toMap(Function.identity(), keyspace -> CqlUtils.extractReplicationFactor((String)fullSchemaSample, (String)keyspace)));
        Assertions.assertThat(rfMap).containsKey((Object)"keyspace");
        Assertions.assertThat(rfMap).containsKey((Object)"quoted_keyspace");
        Assertions.assertThat((Integer)((Integer)rfMap.get("keyspace").getOptions().get("datacenter1"))).isEqualTo(4);
        Assertions.assertThat((Integer)((Integer)rfMap.get("keyspace").getOptions().get("datacenter2"))).isEqualTo(3);
        Assertions.assertThat((Integer)((Integer)rfMap.get("quoted_keyspace").getOptions().get("datacenter1"))).isEqualTo(3);
        Assertions.assertThat((Integer)((Integer)rfMap.get("quoted_keyspace").getOptions().get("datacenter2"))).isEqualTo(3);
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.cassandra.spark.data.VersionRunner#bridges"})
    public void testExtractReplicationFactor(CassandraBridge bridge) {
        ReplicationFactor keyspaceRf = CqlUtils.extractReplicationFactor((String)fullSchemaSample, (String)"keyspace");
        Assertions.assertThat((Object)keyspaceRf).isNotNull();
        Assertions.assertThat((Comparable)keyspaceRf.getReplicationStrategy()).isEqualTo((Object)ReplicationFactor.ReplicationStrategy.NetworkTopologyStrategy);
        Assertions.assertThat((Integer)keyspaceRf.getTotalReplicationFactor()).isEqualTo(7);
        Assertions.assertThat((Map)keyspaceRf.getOptions()).isEqualTo((Object)ImmutableMap.of((Object)"datacenter1", (Object)4, (Object)"datacenter2", (Object)3));
        ReplicationFactor tagEntityRelationV4Rf = CqlUtils.extractReplicationFactor((String)fullSchemaSample, (String)"quoted_keyspace");
        Assertions.assertThat((Object)tagEntityRelationV4Rf).isNotNull();
        Assertions.assertThat((Comparable)tagEntityRelationV4Rf.getReplicationStrategy()).isEqualTo((Object)ReplicationFactor.ReplicationStrategy.NetworkTopologyStrategy);
        Assertions.assertThat((Integer)tagEntityRelationV4Rf.getTotalReplicationFactor()).isEqualTo(6);
        Assertions.assertThat((Map)tagEntityRelationV4Rf.getOptions()).isEqualTo((Object)ImmutableMap.of((Object)"datacenter1", (Object)3, (Object)"datacenter2", (Object)3));
        ReplicationFactor systemDistributedRf = CqlUtils.extractReplicationFactor((String)fullSchemaSample, (String)"system_distributed");
        Assertions.assertThat((Object)systemDistributedRf).isNotNull();
        Assertions.assertThat((Comparable)systemDistributedRf.getReplicationStrategy()).isEqualTo((Object)ReplicationFactor.ReplicationStrategy.SimpleStrategy);
        Assertions.assertThat((Integer)systemDistributedRf.getTotalReplicationFactor()).isEqualTo(3);
        Assertions.assertThat((Map)systemDistributedRf.getOptions()).isEqualTo((Object)ImmutableMap.of((Object)"replication_factor", (Object)3));
        ReplicationFactor systemSchemaRf = CqlUtils.extractReplicationFactor((String)fullSchemaSample, (String)"system_schema");
        Assertions.assertThat((Object)systemSchemaRf).isNotNull();
        Assertions.assertThat((Comparable)systemSchemaRf.getReplicationStrategy()).isEqualTo((Object)ReplicationFactor.ReplicationStrategy.LocalStrategy);
        Assertions.assertThat((Integer)systemSchemaRf.getTotalReplicationFactor()).isEqualTo(0);
        Assertions.assertThat((Map)systemSchemaRf.getOptions()).isEqualTo((Object)ImmutableMap.of());
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.cassandra.spark.data.VersionRunner#bridges"})
    public void testEscapedColumnNames(CassandraBridge bridge) {
        String cleaned = CqlUtils.extractTableSchema((String)fullSchemaSample, (String)"cycling", (String)"rank_by_year_and_name_quoted_columns");
        Assertions.assertThat((String)cleaned).isEqualTo("CREATE TABLE cycling.rank_by_year_and_name_quoted_columns(    race_year      int,    \"RACE_NAME\"    text,    rank           int,    \"cyclist_Name\" text,    PRIMARY KEY ((race_year, \"RACE_NAME\"), rank)) WITH CLUSTERING ORDER BY (rank ASC) AND bloom_filter_fp_chance = 0.01 AND compression = { 'chunk_length_in_kb' : 16, 'class' : 'org.apache.cassandra.io.compress.LZ4Compressor' } AND default_time_to_live = 0 AND min_index_interval = 128 AND max_index_interval = 2048;");
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.cassandra.spark.data.VersionRunner#bridges"})
    public void testExtractTableSchemaCase1(CassandraBridge bridge) {
        String schemaStr = "CREATE TABLE keyspace.table (key blob, column1 text, \"C0\" counter static, \"C1\" counter static, \"C2\" counter static, \"C3\" counter static, \"C4\" counter static, value counter, PRIMARY KEY (key, column1) ) WITH COMPACT STORAGE AND CLUSTERING ORDER BY (column1 ASC) AND bloom_filter_fp_chance = 0.1 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'} AND comment = '' AND compaction = {'class': 'org.apache.cassandra.db.compaction.LeveledCompactionStrategy'} AND compression = {'chunk_length_in_kb': '64', 'class': 'org.apache.cassandra.io.compress.DeflateCompressor'} AND crc_check_chance = 1.0 AND dclocal_read_repair_chance = 0.1 AND default_time_to_live = 0 AND gc_grace_seconds = 864000 AND max_index_interval = 2048 AND memtable_flush_period_in_ms = 0 AND min_index_interval = 128  AND read_repair_chance = 0.0 AND speculative_retry = '99p';";
        String expectedCreateStmt = "CREATE TABLE keyspace.table (key blob, column1 text, \"C0\" counter static, \"C1\" counter static, \"C2\" counter static, \"C3\" counter static, \"C4\" counter static, value counter, PRIMARY KEY (key, column1) ) WITH CLUSTERING ORDER BY (column1 ASC) AND bloom_filter_fp_chance = 0.1 AND compression = {'chunk_length_in_kb': '64', 'class': 'org.apache.cassandra.io.compress.DeflateCompressor'} AND default_time_to_live = 0 AND max_index_interval = 2048 AND min_index_interval = 128;";
        String actualCreateStmt = CqlUtils.extractTableSchema((String)schemaStr, (String)"keyspace", (String)"table");
        Assertions.assertThat((String)actualCreateStmt).isEqualTo(expectedCreateStmt);
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.cassandra.spark.data.VersionRunner#bridges"})
    public void testFailsWithUnbalancedParenthesis(CassandraBridge bridge) {
        String schemaStr = "CREATE TABLE keyspace.table (key blob, c0 text, c1 text, PRIMARY KEY (key);";
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> CqlUtils.extractTableSchema((String)schemaStr, (String)"keyspace", (String)"table")).isInstanceOf(RuntimeException.class)).hasMessage("Found unbalanced parentheses in table schema CREATE TABLE keyspace.table (key blob, c0 text, c1 text, PRIMARY KEY (key);");
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.cassandra.spark.data.VersionRunner#bridges"})
    public void testExtractTableSchemaCase2(CassandraBridge bridge) {
        String schemaStr = "CREATE TABLE keyspace.table (key blob, column1 text, \"C0\" blob, \"C1\" blob, \"C2\" blob, \"C4\" blob, value counter, PRIMARY KEY (key, column1) ) WITH bloom_filter_fp_chance = 0.1 AND cdc = false AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'} AND comment = '' AND compaction = {'class': 'org.apache.cassandra.db.compaction.LeveledCompactionStrategy'} AND compression = {'chunk_length_in_kb': '64', 'class': 'org.apache.cassandra.io.compress.DeflateCompressor'} AND crc_check_chance = 1.0 AND dclocal_read_repair_chance = 0.1 AND default_time_to_live = 100 AND gc_grace_seconds = 864000 AND max_index_interval = 2048 AND memtable_flush_period_in_ms = 0 AND min_index_interval = 128 AND read_repair_chance = 0.0 AND speculative_retry = '99p';";
        String expectedCreateStmt = "CREATE TABLE keyspace.table (key blob, column1 text, \"C0\" blob, \"C1\" blob, \"C2\" blob, \"C4\" blob, value counter, PRIMARY KEY (key, column1) ) WITH bloom_filter_fp_chance = 0.1 AND compression = {'chunk_length_in_kb': '64', 'class': 'org.apache.cassandra.io.compress.DeflateCompressor'} AND default_time_to_live = 100 AND max_index_interval = 2048 AND min_index_interval = 128;";
        String actualCreateStmt = CqlUtils.extractTableSchema((String)schemaStr, (String)"keyspace", (String)"table");
        Assertions.assertThat((String)actualCreateStmt).isEqualTo(expectedCreateStmt);
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.cassandra.spark.data.VersionRunner#bridges"})
    public void testEscapedTableName(CassandraBridge bridge) {
        String schemaStr = "CREATE TABLE ks.\\\"tb\\\" (\\n\\\"key\\\" text,\\n \\\"id1\\\" text,\\n \\\"id2\\\" text,\\n \\\"id3\\\" text,\\n created timestamp,\\n id4 uuid,\\n metadata blob,\\n PRIMARY KEY ((\\\"key\\\", \\\"id1\\\"), \\\"id2\\\", \\\"id3\\\")\\n) WITH CLUSTERING ORDER BY (\\\"id2\\\" DESC, \\\"id3\\\" ASC)\\n    AND read_repair_chance = 0.0\\n    AND dclocal_read_repair_chance = 0.1\\n    AND gc_grace_seconds = 864000\\n    AND bloom_filter_fp_chance = 0.1\\n    AND caching = { 'keys' : 'ALL', 'rows_per_partition' : 'NONE' }\\n    AND comment = ''\\n    AND compaction = { 'class' : 'org.apache.cassandra.db.compaction.LeveledCompactionStrategy', 'max_threshold' : 32, 'min_threshold' : 4 }\\n    AND compression = { 'chunk_length_in_kb' : 64, 'class' : 'org.apache.cassandra.io.compress.LZ4Compressor' }\\n    AND default_time_to_live = 0\\n    AND speculative_retry = '99p'\\n    AND min_index_interval = 128\\n    AND max_index_interval = 2048\\n    AND crc_check_chance = 1.0\\n    AND memtable_flush_period_in_ms = 0;";
        String expectedCreateStmt = "CREATE TABLE ks.\"tb\" (\"key\" text, \"id1\" text, \"id2\" text, \"id3\" text, created timestamp, id4 uuid, metadata blob, PRIMARY KEY ((\"key\", \"id1\"), \"id2\", \"id3\")) WITH CLUSTERING ORDER BY (\"id2\" DESC, \"id3\" ASC) AND bloom_filter_fp_chance = 0.1 AND compression = { 'chunk_length_in_kb' : 64, 'class' : 'org.apache.cassandra.io.compress.LZ4Compressor' } AND default_time_to_live = 0 AND min_index_interval = 128 AND max_index_interval = 2048;";
        String actualCreateStmt = CqlUtils.extractTableSchema((String)schemaStr, (String)"ks", (String)"tb");
        Assertions.assertThat((String)actualCreateStmt).isEqualTo(expectedCreateStmt);
        CqlTable table = bridge.buildSchema(actualCreateStmt, "ks", new ReplicationFactor(ReplicationFactor.ReplicationStrategy.NetworkTopologyStrategy, (Map)ImmutableMap.of((Object)"datacenter1", (Object)3)), Partitioner.Murmur3Partitioner, Collections.emptySet(), null, 0, false);
        Assertions.assertThat((String)table.keyspace()).isEqualTo("ks");
        Assertions.assertThat((String)table.table()).isEqualTo("tb");
        Assertions.assertThat((String)table.getField("key").name()).isEqualTo("key");
        Assertions.assertThat((String)table.getField("id1").name()).isEqualTo("id1");
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.cassandra.spark.data.VersionRunner#bridges"})
    public void testExtractTableSchemaCase3(CassandraBridge bridge) {
        String schemaStr = "CREATE TABLE keyspace.table (key blob, column1 text, \"C0\" blob, \"C1\" blob, \"C2\" blob, \"C4\" blob, value counter, PRIMARY KEY ((key, column1), value) ) WITH bloom_filter_fp_chance = 0.1 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'} AND comment = '' AND compaction = {'class': 'org.apache.cassandra.db.compaction.LeveledCompactionStrategy'} AND compression = {'chunk_length_in_kb': '64', 'class': 'org.apache.cassandra.io.compress.DeflateCompressor'} AND crc_check_chance = 1.0 AND dclocal_read_repair_chance = 0.1 AND default_time_to_live = 0 AND gc_grace_seconds = 864000 AND max_index_interval = 2048 AND memtable_flush_period_in_ms = 0 AND min_index_interval = 128 AND read_repair_chance = 0.0 AND speculative_retry = '99p';";
        String expectedCreateStmt = "CREATE TABLE keyspace.table (key blob, column1 text, \"C0\" blob, \"C1\" blob, \"C2\" blob, \"C4\" blob, value counter, PRIMARY KEY ((key, column1), value) ) WITH bloom_filter_fp_chance = 0.1 AND compression = {'chunk_length_in_kb': '64', 'class': 'org.apache.cassandra.io.compress.DeflateCompressor'} AND default_time_to_live = 0 AND max_index_interval = 2048 AND min_index_interval = 128;";
        String actualCreateStmt = CqlUtils.extractTableSchema((String)schemaStr, (String)"keyspace", (String)"table");
        Assertions.assertThat((String)actualCreateStmt).isEqualTo(expectedCreateStmt);
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.cassandra.spark.data.VersionRunner#bridges"})
    public void testBasicExtractUDTs(CassandraBridge bridge) {
        String schemaStr = "CREATE TYPE udt_keyspace.test_idt1 (a text, b bigint, c int, d float);\nCREATE TYPE udt_keyspace.test_idt2 (x boolean, y timestamp, z timeuuid);";
        Set udts = CqlUtils.extractUdts((String)schemaStr, (String)"udt_keyspace");
        Assertions.assertThat((Collection)udts).hasSize(2);
        Assertions.assertThat((Collection)udts).contains((Object[])new String[]{"CREATE TYPE udt_keyspace.test_idt1 (a text, b bigint, c int, d float);"});
        Assertions.assertThat((Collection)udts).contains((Object[])new String[]{"CREATE TYPE udt_keyspace.test_idt2 (x boolean, y timestamp, z timeuuid);"});
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.cassandra.spark.data.VersionRunner#bridges"})
    public void testExtractUDTs(CassandraBridge bridge) {
        String schema = "\"CREATE TYPE some_keyspace.udt123 (\\n\" +\n                              \"    x uuid,\\n\" +\n                              \"    y text,\\n\" +\n                              \"    z uuid\\n\" +\n                              \");\\n\" +\"CREATE KEYSPACE udt_keyspace WITH REPLICATION = { 'class' : 'org.apache.cassandra.locator.NetworkTopologyStrategy', 'datacenter1': '3' } AND DURABLE_WRITES = true;\nCREATE TYPE udt_keyspace.type_with_time_zone (\n    time bigint,\n    \\\"timezoneOffsetMinutes\\\" int\n);\nCREATE TYPE udt_keyspace.type_1 (\n    \\\"x\\\" text,\n    \\\"y\\\" text,\n    z text,\n    \\\"a\\\" boolean\n);\nCREATE TYPE udt_keyspace.field_with_timestamp (\n    field text,\n    \\\"timeWithZone\\\" frozen<udt_keyspace.type_with_time_zone>\n);\nCREATE TYPE udt_keyspace.type_with_frozen_fields (\n    \\\"f1\\\" frozen<udt_keyspace.field_with_timestamp>,\n    \\\"f2\\\" frozen<udt_keyspace.field_with_timestamp>,\n    \\\"f3\\\" frozen<udt_keyspace.field_with_timestamp>,\n    \\\"f4\\\" frozen<udt_keyspace.field_with_timestamp>,\n    \\\"f5\\\" frozen<udt_keyspace.field_with_timestamp>,\n    \\\"f6\\\" frozen<udt_keyspace.field_with_timestamp>,\n    \\\"f7\\\" frozen<udt_keyspace.field_with_timestamp>,\n    \\\"f8\\\" frozen<udt_keyspace.field_with_timestamp>,\n    \\\"f9\\\" text,\n    \\\"f10\\\" frozen<map<bigint, frozen<map<text, boolean>>>>\n);\nCREATE TYPE another_keyspace.some_udt (\n    x uuid,\n    y text,\n    z uuid\n);\nCREATE TYPE another_keyspace.another_udt (\n    a uuid,\n    b text,\n    c uuid\n);";
        Set udts = CqlUtils.extractUdts((String)schema, (String)"udt_keyspace");
        Assertions.assertThat((Collection)udts).hasSize(4);
        Assertions.assertThat((Collection)udts).contains((Object[])new String[]{"CREATE TYPE udt_keyspace.type_with_frozen_fields (    \"f1\" frozen<udt_keyspace.field_with_timestamp>,    \"f2\" frozen<udt_keyspace.field_with_timestamp>,    \"f3\" frozen<udt_keyspace.field_with_timestamp>,    \"f4\" frozen<udt_keyspace.field_with_timestamp>,    \"f5\" frozen<udt_keyspace.field_with_timestamp>,    \"f6\" frozen<udt_keyspace.field_with_timestamp>,    \"f7\" frozen<udt_keyspace.field_with_timestamp>,    \"f8\" frozen<udt_keyspace.field_with_timestamp>,    \"f9\" text,    \"f10\" frozen<map<bigint, frozen<map<text, boolean>>>>);"});
        Assertions.assertThat((Collection)udts).contains((Object[])new String[]{"CREATE TYPE udt_keyspace.type_with_time_zone (    time bigint,    \"timezoneOffsetMinutes\" int);"});
        Assertions.assertThat((Collection)udts).contains((Object[])new String[]{"CREATE TYPE udt_keyspace.type_1 (    \"x\" text,    \"y\" text,    z text,    \"a\" boolean);"});
        Assertions.assertThat((Collection)udts).contains((Object[])new String[]{"CREATE TYPE udt_keyspace.field_with_timestamp (    field text,    \"timeWithZone\" frozen<udt_keyspace.type_with_time_zone>);"});
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.cassandra.spark.data.VersionRunner#bridges"})
    public void testExtractKeyspacesUDTs(CassandraBridge bridge) {
        String schemaTxt = "\"CREATE KEYSPACE keyspace_with_udts WITH REPLICATION = { 'class' : 'org.apache.cassandra.locator.NetworkTopologyStrategy', 'datacenter1': '3' } AND DURABLE_WRITES = true;\\n\\nCREATE TYPE keyspace_with_udts.type_with_time_zone (\\n    time bigint,\\n    \\\"timezoneOffsetMinutes\\\" int\\n);CREATE TYPE keyspace_with_udts.field_with_timestamp (\\n    field text,\\n    \\\"timeWithZone\\\" frozen<keyspace_with_udts.type_with_time_zone>\\n);\\n\\nCREATE TYPE keyspace_with_udts.type_with_frozen_fields (\\n    \\\"f1\\\" frozen<keyspace_with_udts.field_with_timestamp>,\\n    \\\"f2\\\" frozen<keyspace_with_udts.field_with_timestamp>,\\n    \\\"f3\\\" frozen<map<bigint, int>>\\n);CREATE KEYSPACE ks1 WITH REPLICATION = { 'class' : 'org.apache.cassandra.locator.NetworkTopologyStrategy', 'datacenter1': '3' } AND DURABLE_WRITES = true;\\n\\nCREATE TYPE ks1.type_with_time_zone (\\n    time bigint,\\n    \\\"timezoneOffsetMinutes\\\" int\\n);\\n\\nCREATE TYPE ks1.type_1 (\\n    \\\"f1\\\" text,\\n    \\\"f2\\\" text,\\n    \\\"f3\\\" text\\n);\\n\\nCREATE TYPE ks1.type_2 (\\n    \\\"f1\\\" text,\\n    \\\"f2\\\" text,\\n    \\\"f3\\\" text,\\n    f4 text\\n);\\n\\nCREATE TYPE ks1.field_with_timestamp (\\n    field text,\\n    \\\"timeWithZone\\\" frozen<ks1.type_with_time_zone>\\n);";
        Set udts = CqlUtils.extractUdts((String)schemaTxt, (String)"ks1");
        Assertions.assertThat((Collection)udts).hasSize(4);
        String udtStr = String.join((CharSequence)"\n", udts);
        Assertions.assertThat((String)udtStr).contains(new CharSequence[]{"ks1.type_with_time_zone"});
        Assertions.assertThat((String)udtStr).contains(new CharSequence[]{"ks1.type_1"});
        Assertions.assertThat((String)udtStr).contains(new CharSequence[]{"ks1.type_2"});
        Assertions.assertThat((String)udtStr).contains(new CharSequence[]{"ks1.field_with_timestamp"});
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.cassandra.spark.data.VersionRunner#bridges"})
    public void testExtractTableSchemaCase4(CassandraBridge bridge) {
        String schemaStr = "CREATE TABLE keyspace.table (value text PRIMARY KEY);";
        String expectedCreateStmt = "CREATE TABLE keyspace.table (value text PRIMARY KEY);";
        String actualCreateStmt = CqlUtils.extractTableSchema((String)schemaStr, (String)"keyspace", (String)"table");
        assert (expectedCreateStmt.equals(actualCreateStmt));
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.cassandra.spark.data.VersionRunner#bridges"})
    public void testParseClusteringKeySchema(CassandraBridge bridge) {
        String schemaTxt = "CREATE TABLE ks1.tb1 (\n    namespace int,\n    user_id text,\n    dc_id int,\n    ping_timestamp timestamp,\n    PRIMARY KEY ((namespace, user_id), dc_id)\n) WITH CLUSTERING ORDER BY (dc_id ASC)\n    AND additional_write_policy = '99p'\n    AND bloom_filter_fp_chance = 0.1\n    AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'}\n    AND cdc = false\n    AND comment = ''\n    AND compaction = {'class': 'org.apache.cassandra.db.compaction.LeveledCompactionStrategy', 'max_threshold': '32', 'min_threshold': '4'}\n    AND compression = {'chunk_length_in_kb': '16', 'class': 'org.apache.cassandra.io.compress.LZ4Compressor'}\n    AND crc_check_chance = 1.0\n    AND default_time_to_live = 0\n    AND extensions = {}\n    AND gc_grace_seconds = 864000\n    AND max_index_interval = 256\n    AND memtable_flush_period_in_ms = 0\n    AND min_index_interval = 64\n    AND read_repair = 'BLOCKING'\n    AND speculative_retry = 'MIN(99p,15ms)';";
        String actualCreateStmt = CqlUtils.extractTableSchema((String)schemaTxt, (String)"ks1", (String)"tb1");
        bridge.buildSchema(actualCreateStmt, "ks1", new ReplicationFactor(ReplicationFactor.ReplicationStrategy.NetworkTopologyStrategy, (Map)ImmutableMap.of((Object)"datacenter1", (Object)3)), Partitioner.Murmur3Partitioner, Collections.emptySet(), null, 0, false);
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.cassandra.spark.data.VersionRunner#bridges"})
    public void testExtractClusteringKey(CassandraBridge bridge) {
        Assertions.assertThat((String)CqlUtils.extractClustering((String)"CREATE TABLE ks1.tb1 (a int, b text, c int, d timestamp, PRIMARY KEY ((a, b), c) WITH CLUSTERING ORDER BY (c ASC) AND additional_write_policy = '99p';")).isEqualTo("CLUSTERING ORDER BY (c ASC)");
        Assertions.assertThat((String)CqlUtils.extractClustering((String)"WITH CLUSTERING ORDER BY (c ASC)")).isEqualTo("CLUSTERING ORDER BY (c ASC)");
        Assertions.assertThat((String)CqlUtils.extractClustering((String)"WITH CLUSTERING ORDER BY (c ASC);")).isEqualTo("CLUSTERING ORDER BY (c ASC)");
        Assertions.assertThat((String)CqlUtils.extractClustering((String)"**** WITH CLUSTERING ORDER BY (c ASC)  AND ****     AND   ******* AND '***';")).isEqualTo("CLUSTERING ORDER BY (c ASC)");
        Assertions.assertThat((String)CqlUtils.extractClustering((String)"CREATE TABLE ks1.tb1 (a int, b text, c int, d timestamp, PRIMARY KEY ((a, b), c) WITH CLUSTERING ORDER BY (a DESC, b ASC, c ASC) AND additional_write_policy = '99p' AND bloom_filter_fp_chance = 0.1 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'} AND cdc = false AND comment = '' AND compaction = {'class': 'org.apache.cassandra.db.compaction.LeveledCompactionStrategy', 'max_threshold': '32', 'min_threshold': '4'} AND compression = {'chunk_length_in_kb': '16', 'class': 'org.apache.cassandra.io.compress.LZ4Compressor'} AND crc_check_chance = 1.0 AND default_time_to_live = 0 AND extensions = {} AND gc_grace_seconds = 864000 AND max_index_interval = 256 AND memtable_flush_period_in_ms = 0 AND min_index_interval = 64 AND read_repair = 'BLOCKING' AND speculative_retry = 'MIN(99p,15ms)';")).isEqualTo("CLUSTERING ORDER BY (a DESC, b ASC, c ASC)");
        Assertions.assertThat((String)CqlUtils.extractClustering((String)"CREATE TABLE ks1.tb1 (a int, b text, c int, d timestamp, PRIMARY KEY ((a, b), c) WITH CLUSTERING ORDER BY (a DESC, b ASC, c ASC) AND additional_write_policy = '99p' AND bloom_filter_fp_chance = 0.1 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'} AND cdc = false AND comment = '' AND compaction = {'class': 'org.apache.cassandra.db.compaction.LeveledCompactionStrategy', 'max_threshold': '32', 'min_threshold': '4'} AND compression = {'chunk_length_in_kb': '16', 'class': 'org.apache.cassandra.io.compress.LZ4Compressor'} AND crc_check_chance = 1.0 AND default_time_to_live = 0 AND extensions = {} AND gc_grace_seconds = 864000 AND max_index_interval = 256 AND memtable_flush_period_in_ms = 0 AND min_index_interval = 64 AND read_repair = 'BLOCKING' AND speculative_retry = 'MIN(99p,15ms)'")).isEqualTo("CLUSTERING ORDER BY (a DESC, b ASC, c ASC)");
        Assertions.assertThat((String)CqlUtils.extractClustering((String)"WITH CLUSTERING ORDER BY (a DESC, b ASC, c ASC)")).isEqualTo("CLUSTERING ORDER BY (a DESC, b ASC, c ASC)");
        Assertions.assertThat((String)CqlUtils.extractClustering((String)"CREATE TABLE ks1.tb1 (a int, b text, c int, d timestamp, PRIMARY KEY ((a, b), c)) WITH CLUSTERING ORDER BY (a ASC) AND speculative_retry = 'MIN(99p,15ms);")).isEqualTo("CLUSTERING ORDER BY (a ASC)");
        Assertions.assertThat((String)CqlUtils.extractClustering((String)"")).isNull();
        Assertions.assertThat((String)CqlUtils.extractClustering((String)"CREATE TABLE ks1.tb1 (a int, b text, c int, d timestamp, PRIMARY KEY ((a, b), c);")).isNull();
        Assertions.assertThat((String)CqlUtils.extractClustering((String)"CREATE TABLE ks1.tb1 (a int, b text, c int, d timestamp, PRIMARY KEY ((a, b), c)")).isNull();
        Assertions.assertThat((String)CqlUtils.extractClustering((String)"CREATE TABLE ks1.tb1 (a int, b text, c int, d timestamp, PRIMARY KEY ((a, b), c) AND additional_write_policy = '99p';")).isNull();
        Assertions.assertThat((String)CqlUtils.extractClustering((String)"CREATE TABLE ks1.tb1 (a int, b text, c int, d timestamp, PRIMARY KEY ((a, b), c) AND additional_write_policy = '99p'")).isNull();
        Assertions.assertThat((String)CqlUtils.extractClustering((String)"CREATE TABLE ks1.tb1 (a int, b text, c int, d timestamp, PRIMARY KEY ((a, b), c) AND additional_write_policy = '99p' AND bloom_filter_fp_chance = 0.1 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'} AND cdc = false AND comment = '' AND compaction = {'class': 'org.apache.cassandra.db.compaction.LeveledCompactionStrategy', 'max_threshold': '32', 'min_threshold': '4'} AND compression = {'chunk_length_in_kb': '16', 'class': 'org.apache.cassandra.io.compress.LZ4Compressor'} AND crc_check_chance = 1.0 AND default_time_to_live = 0 AND extensions = {} AND gc_grace_seconds = 864000 AND max_index_interval = 256 AND memtable_flush_period_in_ms = 0 AND min_index_interval = 64 AND read_repair = 'BLOCKING' AND speculative_retry = 'MIN(99p,15ms)';")).isNull();
        Assertions.assertThat((String)CqlUtils.extractClustering((String)"CREATE TABLE ks1.tb1 (a int, b text, c int, d timestamp, PRIMARY KEY ((a, b), c) AND additional_write_policy = '99p' AND bloom_filter_fp_chance = 0.1 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'} AND cdc = false AND comment = '' AND compaction = {'class': 'org.apache.cassandra.db.compaction.LeveledCompactionStrategy', 'max_threshold': '32', 'min_threshold': '4'} AND compression = {'chunk_length_in_kb': '16', 'class': 'org.apache.cassandra.io.compress.LZ4Compressor'} AND crc_check_chance = 1.0 AND default_time_to_live = 0 AND extensions = {} AND gc_grace_seconds = 864000 AND max_index_interval = 256 AND memtable_flush_period_in_ms = 0 AND min_index_interval = 64 AND read_repair = 'BLOCKING' AND speculative_retry = 'MIN(99p,15ms)'")).isNull();
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.cassandra.spark.data.VersionRunner#bridges"})
    public void testClusteringOrderByIsRetained(CassandraBridge bridge) {
        String schemaStr = "CREATE TABLE keyspace.table (id bigint, version bigint PRIMARY KEY (id, version)) WITH CLUSTERING ORDER BY (id DESC, version DESC) AND foo = 1;";
        String expectedCreateStmt = "CREATE TABLE keyspace.table (id bigint, version bigint PRIMARY KEY (id, version)) WITH CLUSTERING ORDER BY (id DESC, version DESC);";
        String actualCreateStmt = CqlUtils.extractTableSchema((String)schemaStr, (String)"keyspace", (String)"table");
        Assertions.assertThat((String)actualCreateStmt).isEqualTo(expectedCreateStmt);
    }

    @Test
    public void testExtractCdcFlag() {
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int);", Collections.singletonList("cdc"))).isEmpty();
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH cdc = true;", Collections.singletonList("cdc"))).contains((Object[])new String[]{"cdc = true"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH cdc = false;", Collections.singletonList("cdc"))).contains((Object[])new String[]{"cdc = false"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH cdc = true AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'};", Collections.singletonList("cdc"))).contains((Object[])new String[]{"cdc = true"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH cdc = false AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'};", Collections.singletonList("cdc"))).contains((Object[])new String[]{"cdc = false"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH bloom_filter_fp_chance = 0.1 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'} AND cdc = true;", Collections.singletonList("cdc"))).contains((Object[])new String[]{"cdc = true"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH bloom_filter_fp_chance = 0.1 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'} AND cdc = false;", Collections.singletonList("cdc"))).contains((Object[])new String[]{"cdc = false"});
    }

    @Test
    public void testExtractDefaultTtlOption() {
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int);", Collections.singletonList("default_time_to_live"))).isEmpty();
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH default_time_to_live = 1;", Collections.singletonList("default_time_to_live"))).contains((Object[])new String[]{"default_time_to_live = 1"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH default_time_to_live = 2;", Collections.singletonList("default_time_to_live"))).contains((Object[])new String[]{"default_time_to_live = 2"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH default_time_to_live = 3 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'};", Collections.singletonList("default_time_to_live"))).contains((Object[])new String[]{"default_time_to_live = 3"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH default_time_to_live = 4 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'};", Collections.singletonList("default_time_to_live"))).contains((Object[])new String[]{"default_time_to_live = 4"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH bloom_filter_fp_chance = 0.1 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'} AND default_time_to_live = 5;", Collections.singletonList("default_time_to_live"))).contains((Object[])new String[]{"default_time_to_live = 5"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH bloom_filter_fp_chance = 0.1 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'} AND default_time_to_live = 6;", Collections.singletonList("default_time_to_live"))).contains((Object[])new String[]{"default_time_to_live = 6"});
    }

    @Test
    public void testExtractBloomFilterFalsePositiveChance() {
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int);", Collections.singletonList("bloom_filter_fp_chance"))).isEmpty();
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH bloom_filter_fp_chance = 0.1;", Collections.singletonList("bloom_filter_fp_chance"))).contains((Object[])new String[]{"bloom_filter_fp_chance = 0.1"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH bloom_filter_fp_chance = 0.2;", Collections.singletonList("bloom_filter_fp_chance"))).contains((Object[])new String[]{"bloom_filter_fp_chance = 0.2"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH bloom_filter_fp_chance = 0.3 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'};", Collections.singletonList("bloom_filter_fp_chance"))).contains((Object[])new String[]{"bloom_filter_fp_chance = 0.3"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH bloom_filter_fp_chance = 0.4 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'};", Collections.singletonList("bloom_filter_fp_chance"))).contains((Object[])new String[]{"bloom_filter_fp_chance = 0.4"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH bloom_filter_fp_chance = 0.1 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'} AND bloom_filter_fp_chance = 0.5;", Collections.singletonList("bloom_filter_fp_chance"))).contains((Object[])new String[]{"bloom_filter_fp_chance = 0.5"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH bloom_filter_fp_chance = 0.1 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'} AND bloom_filter_fp_chance = 0.6;", Collections.singletonList("bloom_filter_fp_chance"))).contains((Object[])new String[]{"bloom_filter_fp_chance = 0.6"});
    }

    @Test
    public void testExtractCompression() {
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int);", Collections.singletonList("compression"))).isEmpty();
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH compression = { 'fake_option': '0.1' };", Collections.singletonList("compression"))).contains((Object[])new String[]{"compression = { 'fake_option': '0.1' }"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH compression = { 'fake_option': '0.2' };", Collections.singletonList("compression"))).contains((Object[])new String[]{"compression = { 'fake_option': '0.2' }"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH compression = { 'fake_option': '0.3' } AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'};", Collections.singletonList("compression"))).contains((Object[])new String[]{"compression = { 'fake_option': '0.3' }"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH compression = { 'fake_option': '0.4' } AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'};", Collections.singletonList("compression"))).contains((Object[])new String[]{"compression = { 'fake_option': '0.4' }"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH bloom_filter_fp_chance = 0.1 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'} AND compression = { 'fake_option': '0.5' };", Collections.singletonList("compression"))).contains((Object[])new String[]{"compression = { 'fake_option': '0.5' }"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH bloom_filter_fp_chance = 0.1 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'}  AND compression = { 'fake_option': '0.6' };", Collections.singletonList("compression"))).contains((Object[])new String[]{"compression = { 'fake_option': '0.6' }"});
    }

    @Test
    public void testExtractMinIndexInterval() {
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int);", Collections.singletonList("min_index_interval"))).isEmpty();
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH min_index_interval = 1;", Collections.singletonList("min_index_interval"))).contains((Object[])new String[]{"min_index_interval = 1"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH min_index_interval = 2;", Collections.singletonList("min_index_interval"))).contains((Object[])new String[]{"min_index_interval = 2"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH min_index_interval = 3 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'};", Collections.singletonList("min_index_interval"))).contains((Object[])new String[]{"min_index_interval = 3"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH min_index_interval = 4 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'};", Collections.singletonList("min_index_interval"))).contains((Object[])new String[]{"min_index_interval = 4"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH bloom_filter_fp_chance = 0.1 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'} AND min_index_interval = 5;", Collections.singletonList("min_index_interval"))).contains((Object[])new String[]{"min_index_interval = 5"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH bloom_filter_fp_chance = 0.1 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'} AND min_index_interval = 6;", Collections.singletonList("min_index_interval"))).contains((Object[])new String[]{"min_index_interval = 6"});
    }

    @Test
    public void testExtractMaxIndexInterval() {
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int);", Collections.singletonList("max_index_interval"))).isEmpty();
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH max_index_interval = 1;", Collections.singletonList("max_index_interval"))).contains((Object[])new String[]{"max_index_interval = 1"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH max_index_interval = 2;", Collections.singletonList("max_index_interval"))).contains((Object[])new String[]{"max_index_interval = 2"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH max_index_interval = 3 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'};", Collections.singletonList("max_index_interval"))).contains((Object[])new String[]{"max_index_interval = 3"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH max_index_interval = 4 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'};", Collections.singletonList("max_index_interval"))).contains((Object[])new String[]{"max_index_interval = 4"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH bloom_filter_fp_chance = 0.1 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'} AND max_index_interval = 5;", Collections.singletonList("max_index_interval"))).contains((Object[])new String[]{"max_index_interval = 5"});
        Assertions.assertThat((List)CqlUtils.extractOverrideProperties((String)"CREATE TABLE k.t (a int PRIMARY KEY, b int) WITH bloom_filter_fp_chance = 0.1 AND caching = {'keys': 'ALL', 'rows_per_partition': 'NONE'} AND max_index_interval = 6;", Collections.singletonList("max_index_interval"))).contains((Object[])new String[]{"max_index_interval = 6"});
    }

    @Test
    public void testCdcExtractSchema() {
        String schema = "CREATE KEYSPACE ks1 WITH REPLICATION = { 'class' : 'org.apache.cassandra.locator.NetworkTopologyStrategy', 'MS': '3', 'ST': '3' } AND DURABLE_WRITES = true;\\n\\nCREATE TABLE ks1.tb1 (\"a\" text, \"b\" text, \"c\" text, \"d\" text, e timestamp, f uuid, g blob, PRIMARY KEY ((\"a\", \"b\"), \"c\", \"d\")) WITH CLUSTERING ORDER BY (\"c\" DESC, \"d\" ASC) AND cdc = true;\n\nCREATE TABLE ks1.tb2 (a text, b text, c text, d text, PRIMARY KEY ((a), b, c)) WITH CLUSTERING ORDER BY (b DESC, c ASC) AND cdc = true;\n\nCREATE KEYSPACE ks2 WITH REPLICATION = { 'class' : 'org.apache.cassandra.locator.NetworkTopologyStrategy', 'MS': '3', 'ST': '3' } AND DURABLE_WRITES = true;\\n\\nCREATE TABLE ks2.tb3 (a text, b text, c text, d text, PRIMARY KEY ((a), b, c)) WITH CLUSTERING ORDER BY (b DESC, c ASC);\n\nCREATE TABLE ks2.tb4 (a bigint, b int, c uuid, d text, PRIMARY KEY (a)) WITH cdc = true;\n\n";
        Map createStmts = CqlUtils.extractCdcTables((String)schema);
        Assertions.assertThat((int)3).isEqualTo(createStmts.size());
        Assertions.assertThat((boolean)createStmts.containsKey(TableIdentifier.of((String)"ks1", (String)"tb1"))).isTrue();
        Assertions.assertThat((boolean)createStmts.containsKey(TableIdentifier.of((String)"ks1", (String)"tb2"))).isTrue();
        Assertions.assertThat((boolean)createStmts.containsKey(TableIdentifier.of((String)"ks2", (String)"tb3"))).isFalse();
        Assertions.assertThat((boolean)createStmts.containsKey(TableIdentifier.of((String)"ks2", (String)"tb4"))).isTrue();
    }

    private static String loadFullSchemaSample() throws IOException {
        Path fullSchemaSampleFile = ResourceUtils.writeResourceToPath(CqlUtilsTest.class.getClassLoader(), tempPath, "cql/fullSchema.cql");
        return FileUtils.readFileToString((File)fullSchemaSampleFile.toFile(), (Charset)StandardCharsets.UTF_8);
    }
}

