/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.server.purgatory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.kafka.server.metrics.KafkaMetricsGroup;
import org.apache.kafka.server.purgatory.DelayedOperation;
import org.apache.kafka.server.purgatory.DelayedOperationKey;
import org.apache.kafka.server.util.ShutdownableThread;
import org.apache.kafka.server.util.timer.SystemTimer;
import org.apache.kafka.server.util.timer.Timer;
import org.apache.kafka.server.util.timer.TimerTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DelayedOperationPurgatory<T extends DelayedOperation> {
    private static final Logger LOG = LoggerFactory.getLogger(DelayedOperationPurgatory.class);
    private static final int SHARDS = 512;
    private final KafkaMetricsGroup metricsGroup = new KafkaMetricsGroup("kafka.server", "DelayedOperationPurgatory");
    private final Map<String, String> metricsTags;
    private final List<WatcherList> watcherLists;
    private final AtomicInteger estimatedTotalOperations = new AtomicInteger(0);
    private final ExpiredOperationReaper expirationReaper = new ExpiredOperationReaper();
    private final String purgatoryName;
    private final Timer timeoutTimer;
    private final int brokerId;
    private final int purgeInterval;
    private final boolean reaperEnabled;
    private final boolean timerEnabled;

    public DelayedOperationPurgatory(String purgatoryName, Timer timer, int brokerId, boolean reaperEnabled) {
        this(purgatoryName, timer, brokerId, 1000, reaperEnabled, true);
    }

    public DelayedOperationPurgatory(String purgatoryName, int brokerId) {
        this(purgatoryName, brokerId, 1000);
    }

    public DelayedOperationPurgatory(String purgatoryName, int brokerId, int purgeInterval) {
        this(purgatoryName, new SystemTimer(purgatoryName), brokerId, purgeInterval, true, true);
    }

    public DelayedOperationPurgatory(String purgatoryName, Timer timeoutTimer, int brokerId, int purgeInterval, boolean reaperEnabled, boolean timerEnabled) {
        this.purgatoryName = purgatoryName;
        this.timeoutTimer = timeoutTimer;
        this.brokerId = brokerId;
        this.purgeInterval = purgeInterval;
        this.reaperEnabled = reaperEnabled;
        this.timerEnabled = timerEnabled;
        this.watcherLists = new ArrayList<WatcherList>(512);
        for (int i = 0; i < 512; ++i) {
            this.watcherLists.add(new WatcherList());
        }
        this.metricsTags = Collections.singletonMap("delayedOperation", purgatoryName);
        this.metricsGroup.newGauge("PurgatorySize", this::watched, this.metricsTags);
        this.metricsGroup.newGauge("NumDelayedOperations", this::numDelayed, this.metricsTags);
        if (reaperEnabled) {
            this.expirationReaper.start();
        }
    }

    private WatcherList watcherList(DelayedOperationKey key) {
        return this.watcherLists.get(Math.abs(key.hashCode() % this.watcherLists.size()));
    }

    public <K extends DelayedOperationKey> boolean tryCompleteElseWatch(T operation, List<K> watchKeys) {
        if (watchKeys.isEmpty()) {
            throw new IllegalArgumentException("The watch key list can't be empty");
        }
        if (((DelayedOperation)operation).safeTryCompleteOrElse(() -> {
            watchKeys.forEach(key -> {
                if (!operation.isCompleted()) {
                    this.watchForOperation((DelayedOperationKey)key, operation);
                }
            });
            if (!watchKeys.isEmpty()) {
                this.estimatedTotalOperations.incrementAndGet();
            }
        })) {
            return true;
        }
        if (!((DelayedOperation)operation).isCompleted()) {
            if (this.timerEnabled) {
                this.timeoutTimer.add((TimerTask)operation);
            }
            if (((DelayedOperation)operation).isCompleted()) {
                ((TimerTask)operation).cancel();
            }
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public <K extends DelayedOperationKey> int checkAndComplete(K key) {
        int numCompleted;
        Watchers watchers;
        WatcherList wl = this.watcherList(key);
        wl.watchersLock.lock();
        try {
            watchers = wl.watchersByKey.get(key);
        }
        finally {
            wl.watchersLock.unlock();
        }
        int n = numCompleted = watchers == null ? 0 : watchers.tryCompleteWatched();
        if (numCompleted > 0) {
            LOG.debug("Request key {} unblocked {} {} operations", new Object[]{key, numCompleted, this.purgatoryName});
        }
        return numCompleted;
    }

    public int watched() {
        int sum = 0;
        for (WatcherList watcherList : this.watcherLists) {
            sum += watcherList.allWatchers().stream().mapToInt(Watchers::countWatched).sum();
        }
        return sum;
    }

    public int numDelayed() {
        return this.timeoutTimer.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<T> cancelForKey(DelayedOperationKey key) {
        WatcherList wl = this.watcherList(key);
        wl.watchersLock.lock();
        try {
            Watchers watchers = wl.watchersByKey.remove(key);
            if (watchers != null) {
                List list = watchers.cancel();
                return list;
            }
            List list = Collections.emptyList();
            return list;
        }
        finally {
            wl.watchersLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void watchForOperation(DelayedOperationKey key, T operation) {
        WatcherList wl = this.watcherList(key);
        wl.watchersLock.lock();
        try {
            Watchers watcher = wl.watchersByKey.computeIfAbsent(key, x$0 -> new Watchers((DelayedOperationKey)x$0));
            watcher.watch(operation);
        }
        finally {
            wl.watchersLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void removeKeyIfEmpty(DelayedOperationKey key, Watchers watchers) {
        WatcherList wl = this.watcherList(key);
        wl.watchersLock.lock();
        try {
            if (wl.watchersByKey.get(key) != watchers) {
                return;
            }
            if (watchers != null && watchers.isEmpty()) {
                wl.watchersByKey.remove(key);
            }
        }
        finally {
            wl.watchersLock.unlock();
        }
    }

    public void shutdown() throws Exception {
        if (this.reaperEnabled) {
            this.expirationReaper.initiateShutdown();
            this.timeoutTimer.add(new TimerTask(0L){

                @Override
                public void run() {
                }
            });
            this.expirationReaper.awaitShutdown();
        }
        this.timeoutTimer.close();
        this.metricsGroup.removeMetric("PurgatorySize", this.metricsTags);
        this.metricsGroup.removeMetric("NumDelayedOperations", this.metricsTags);
    }

    private void advanceClock(long timeoutMs) throws InterruptedException {
        this.timeoutTimer.advanceClock(timeoutMs);
        if (this.estimatedTotalOperations.get() - this.numDelayed() > this.purgeInterval) {
            this.estimatedTotalOperations.getAndSet(this.numDelayed());
            LOG.debug("Begin purging watch lists");
            int purged = 0;
            for (WatcherList watcherList : this.watcherLists) {
                purged += watcherList.allWatchers().stream().mapToInt(Watchers::purgeCompleted).sum();
            }
            LOG.debug("Purged {} elements from watch lists.", (Object)purged);
        }
    }

    private class ExpiredOperationReaper
    extends ShutdownableThread {
        ExpiredOperationReaper() {
            super("ExpirationReaper-" + DelayedOperationPurgatory.this.brokerId + "-" + DelayedOperationPurgatory.this.purgatoryName, false);
        }

        @Override
        public void doWork() {
            try {
                DelayedOperationPurgatory.this.advanceClock(200L);
            }
            catch (InterruptedException ie) {
                throw new RuntimeException(ie);
            }
        }
    }

    private class WatcherList {
        private final ConcurrentHashMap<DelayedOperationKey, Watchers> watchersByKey = new ConcurrentHashMap();
        private final ReentrantLock watchersLock = new ReentrantLock();

        private WatcherList() {
        }

        Collection<Watchers> allWatchers() {
            return this.watchersByKey.values();
        }
    }

    private class Watchers {
        private final ConcurrentLinkedQueue<T> operations = new ConcurrentLinkedQueue();
        private final DelayedOperationKey key;

        Watchers(DelayedOperationKey key) {
            this.key = key;
        }

        int countWatched() {
            return this.operations.size();
        }

        boolean isEmpty() {
            return this.operations.isEmpty();
        }

        void watch(T t) {
            this.operations.add(t);
        }

        int tryCompleteWatched() {
            int completed = 0;
            Iterator iter = this.operations.iterator();
            while (iter.hasNext()) {
                DelayedOperation curr = (DelayedOperation)iter.next();
                if (curr.isCompleted()) {
                    iter.remove();
                    continue;
                }
                if (!curr.safeTryComplete()) continue;
                iter.remove();
                ++completed;
            }
            if (this.operations.isEmpty()) {
                DelayedOperationPurgatory.this.removeKeyIfEmpty(this.key, this);
            }
            return completed;
        }

        List<T> cancel() {
            Iterator iter = this.operations.iterator();
            ArrayList<DelayedOperation> cancelled = new ArrayList<DelayedOperation>();
            while (iter.hasNext()) {
                DelayedOperation curr = (DelayedOperation)iter.next();
                curr.cancel();
                iter.remove();
                cancelled.add(curr);
            }
            return cancelled;
        }

        int purgeCompleted() {
            int purged = 0;
            Iterator iter = this.operations.iterator();
            while (iter.hasNext()) {
                DelayedOperation curr = (DelayedOperation)iter.next();
                if (!curr.isCompleted()) continue;
                iter.remove();
                ++purged;
            }
            if (this.operations.isEmpty()) {
                DelayedOperationPurgatory.this.removeKeyIfEmpty(this.key, this);
            }
            return purged;
        }
    }
}

